<?php
/**
 * Admin Settings Page
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class LLMSCBT_Admin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_notices', array($this, 'show_notices'));
    }
    
    /**
     * Add menu page
     */
    public function add_menu_page() {
        add_menu_page(
            __('LLMS Central', 'llms-central-ai-bot-tracker'),
            __('AI Bot Tracker', 'llms-central-ai-bot-tracker'),
            'manage_options',
            'llmscbt',
            array($this, 'render_settings_page'),
            'dashicons-visibility',
            100
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('llmscbt_settings', 'llmscbt_tracking_enabled', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean'
        ));
        register_setting('llmscbt_settings', 'llmscbt_domain', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('llmscbt_settings', 'llmscbt_llms_txt_enabled', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean'
        ));
        register_setting('llmscbt_settings', 'llmscbt_llms_txt_content', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field'
        ));
        register_setting('llmscbt_settings', 'llmscbt_consent_given', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean'
        ));
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_assets($hook) {
        if ('toplevel_page_llmscbt' !== $hook) {
            return;
        }
        
        wp_enqueue_style(
            'llmscbt-admin',
            LLMSCBT_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            LLMSCBT_VERSION
        );
        
        wp_enqueue_script(
            'llmscbt-admin',
            LLMSCBT_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            LLMSCBT_VERSION,
            true
        );
    }
    
    /**
     * Show admin notices
     */
    public function show_notices() {
        if (!get_option('llmscbt_consent_given') && get_option('llmscbt_tracking_enabled')) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <?php esc_html_e('LLMS Central Bot Tracker: Please review and accept the privacy policy in the settings to enable tracking.', 'llms-central-ai-bot-tracker'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=llmscbt')); ?>"><?php esc_html_e('Go to Settings', 'llms-central-ai-bot-tracker'); ?></a>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Handle form submission
        if (isset($_POST['llmscbt_save_settings'])) {
            check_admin_referer('llmscbt_settings');
            $this->save_settings();
        }
        
        $tracking_enabled = get_option('llmscbt_tracking_enabled', false);
        $domain = get_option('llmscbt_domain', parse_url(home_url(), PHP_URL_HOST));
        $llms_txt_enabled = get_option('llmscbt_llms_txt_enabled', false);
        $llms_txt_content = get_option('llmscbt_llms_txt_content', '');
        $consent_given = get_option('llmscbt_consent_given', false);
        
        ?>
        <div class="wrap llms-central-admin">
            <h1><?php esc_html_e('LLMS Central - AI Bot Tracker', 'llms-central-ai-bot-tracker'); ?></h1>
            
            <div class="llms-central-header">
                <p class="description">
                    <?php esc_html_e('Track AI bots (GPT, Claude, Gemini) visiting your site and manage your llms.txt file.', 'llms-central-ai-bot-tracker'); ?>
                    <a href="https://llmscentral.com" target="_blank"><?php esc_html_e('Learn more', 'llms-central-ai-bot-tracker'); ?></a>
                </p>
            </div>
            
            <form method="post" action="">
                <?php wp_nonce_field('llmscbt_settings'); ?>
                
                <div class="llms-central-tabs">
                    <nav class="nav-tab-wrapper">
                        <a href="#tracking" class="nav-tab nav-tab-active"><?php esc_html_e('Bot Tracking', 'llms-central-ai-bot-tracker'); ?></a>
                        <a href="#llms-txt" class="nav-tab"><?php esc_html_e('llms.txt File', 'llms-central-ai-bot-tracker'); ?></a>
                        <a href="#privacy" class="nav-tab"><?php esc_html_e('Privacy & Consent', 'llms-central-ai-bot-tracker'); ?></a>
                    </nav>
                    
                    <!-- Bot Tracking Tab -->
                    <div id="tracking" class="tab-content active">
                        <h2><?php esc_html_e('AI Bot Tracking', 'llms-central-ai-bot-tracker'); ?></h2>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Enable Tracking', 'llms-central-ai-bot-tracker'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="llmscbt_tracking_enabled" value="1" <?php checked($tracking_enabled, true); ?>>
                                        <?php esc_html_e('Track AI bots visiting your site', 'llms-central-ai-bot-tracker'); ?>
                                    </label>
                                    <p class="description">
                                        <?php esc_html_e('Tracks GPTBot, Claude, Gemini, and 14+ other AI crawlers. View analytics in your LLMS Central dashboard.', 'llms-central-ai-bot-tracker'); ?>
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('Your Domain', 'llms-central-ai-bot-tracker'); ?></th>
                                <td>
                                    <input type="text" name="llmscbt_domain" value="<?php echo esc_attr($domain); ?>" class="regular-text" readonly>
                                    <p class="description">
                                        <?php esc_html_e('This is your site domain. It will be used to identify your tracking data.', 'llms-central-ai-bot-tracker'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        
                        <?php if ($tracking_enabled && $consent_given): ?>
                        <div class="llms-central-tracking-code">
                            <h3><?php esc_html_e('Tracking Status', 'llms-central-ai-bot-tracker'); ?></h3>
                            <div class="notice notice-success inline">
                                <p>
                                    <span class="dashicons dashicons-yes-alt"></span>
                                    <?php esc_html_e('Tracking is active! AI bot visits are being recorded.', 'llms-central-ai-bot-tracker'); ?>
                                </p>
                            </div>
                            <p>
                                <a href="https://llmscentral.com/dashboard" target="_blank" class="button button-primary">
                                    <?php esc_html_e('View Analytics Dashboard', 'llms-central-ai-bot-tracker'); ?>
                                </a>
                            </p>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- llms.txt Tab -->
                    <div id="llms-txt" class="tab-content">
                        <h2><?php esc_html_e('llms.txt File Management', 'llms-central-ai-bot-tracker'); ?></h2>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Enable llms.txt', 'llms-central-ai-bot-tracker'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="llmscbt_llms_txt_enabled" value="1" <?php checked($llms_txt_enabled, true); ?>>
                                        <?php esc_html_e('Create and manage llms.txt file', 'llms-central-ai-bot-tracker'); ?>
                                    </label>
                                    <p class="description">
                                        <?php esc_html_e('Creates a llms.txt file at your site root to tell AI systems how they can use your content.', 'llms-central-ai-bot-tracker'); ?>
                                    </p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row"><?php esc_html_e('llms.txt Content', 'llms-central-ai-bot-tracker'); ?></th>
                                <td>
                                    <textarea name="llmscbt_llms_txt_content" rows="15" class="large-text code"><?php echo esc_textarea($llms_txt_content); ?></textarea>
                                    <p class="description">
                                        <?php esc_html_e('Edit your llms.txt file content. This tells AI systems your training data policies.', 'llms-central-ai-bot-tracker'); ?>
                                        <a href="https://llmscentral.com/create-llms-txt" target="_blank"><?php esc_html_e('Use our generator', 'llms-central-ai-bot-tracker'); ?></a>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        
                        <?php if ($llms_txt_enabled): ?>
                        <div class="llms-central-file-info">
                            <h3><?php esc_html_e('File Location', 'llms-central-ai-bot-tracker'); ?></h3>
                            <p>
                                <code><?php echo esc_html(home_url('/llms.txt')); ?></code>
                                <a href="<?php echo esc_url(home_url('/llms.txt')); ?>" target="_blank" class="button button-secondary">
                                    <?php esc_html_e('View File', 'llms-central-ai-bot-tracker'); ?>
                                </a>
                            </p>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Privacy Tab -->
                    <div id="privacy" class="tab-content">
                        <h2><?php esc_html_e('Privacy & Data Consent', 'llms-central-ai-bot-tracker'); ?></h2>
                        
                        <div class="llms-central-privacy-notice">
                            <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 12px; margin-bottom: 20px;">
                                <strong><?php esc_html_e('Data Collection is OPT-IN ONLY', 'llms-central-ai-bot-tracker'); ?></strong>
                                <p style="margin: 8px 0 0 0;"><?php esc_html_e('No data is sent to external servers unless you explicitly enable tracking and provide consent below.', 'llms-central-ai-bot-tracker'); ?></p>
                            </div>
                            
                            <h3><?php esc_html_e('How This Plugin Works', 'llms-central-ai-bot-tracker'); ?></h3>
                            <ul>
                                <li><?php esc_html_e('Adds a lightweight tracking script to your site', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Detects AI bot visits (GPTBot, Claude, Gemini, etc.)', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Sends anonymous analytics to LLMS Central servers (only when enabled)', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('No personal data or visitor information is collected', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Only tracks bot user-agents and page URLs', 'llms-central-ai-bot-tracker'); ?></li>
                            </ul>
                            
                            <h3><?php esc_html_e('Data Collected', 'llms-central-ai-bot-tracker'); ?></h3>
                            <ul>
                                <li><?php esc_html_e('Bot user-agent strings (e.g., "GPTBot/1.0")', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Page URLs visited by bots', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Timestamp of bot visits', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Your domain name', 'llms-central-ai-bot-tracker'); ?></li>
                            </ul>
                            
                            <h3><?php esc_html_e('Data NOT Collected', 'llms-central-ai-bot-tracker'); ?></h3>
                            <ul>
                                <li><?php esc_html_e('Visitor IP addresses', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Personal information', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Cookies or tracking identifiers', 'llms-central-ai-bot-tracker'); ?></li>
                                <li><?php esc_html_e('Form submissions or user interactions', 'llms-central-ai-bot-tracker'); ?></li>
                            </ul>
                        </div>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Consent', 'llms-central-ai-bot-tracker'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="llmscbt_consent_given" value="1" <?php checked($consent_given, true); ?> required>
                                        <strong><?php esc_html_e('I understand and consent to the data collection described above', 'llms-central-ai-bot-tracker'); ?></strong>
                                    </label>
                                    <p class="description">
                                        <?php esc_html_e('Required to enable tracking. You can revoke consent at any time by disabling the plugin.', 'llms-central-ai-bot-tracker'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        
                        <p>
                            <a href="https://llmscentral.com/privacy" target="_blank"><?php esc_html_e('View Full Privacy Policy', 'llms-central-ai-bot-tracker'); ?></a> |
                            <a href="https://llmscentral.com/terms" target="_blank"><?php esc_html_e('Terms of Service', 'llms-central-ai-bot-tracker'); ?></a>
                        </p>
                    </div>
                </div>
                
                <p class="submit">
                    <input type="submit" name="llmscbt_save_settings" class="button button-primary" value="<?php esc_attr_e('Save Settings', 'llms-central-ai-bot-tracker'); ?>">
                </p>
            </form>
            
            <div class="llms-central-footer">
                <p>
                    <?php esc_html_e('Need help?', 'llms-central-ai-bot-tracker'); ?>
                    <a href="https://llmscentral.com/wordpress-plugin" target="_blank"><?php esc_html_e('Documentation', 'llms-central-ai-bot-tracker'); ?></a> |
                    <a href="https://llmscentral.com/getting-started" target="_blank"><?php esc_html_e('Getting Started', 'llms-central-ai-bot-tracker'); ?></a> |
                    <a href="https://llmscentral.com/dashboard" target="_blank"><?php esc_html_e('Dashboard', 'llms-central-ai-bot-tracker'); ?></a>
                </p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Save settings
     */
    private function save_settings() {
        update_option('llmscbt_tracking_enabled', isset($_POST['llmscbt_tracking_enabled']));
        update_option('llmscbt_domain', sanitize_text_field($_POST['llmscbt_domain']));
        update_option('llmscbt_llms_txt_enabled', isset($_POST['llmscbt_llms_txt_enabled']));
        update_option('llmscbt_llms_txt_content', sanitize_textarea_field($_POST['llmscbt_llms_txt_content']));
        update_option('llmscbt_consent_given', isset($_POST['llmscbt_consent_given']));
        
        // Create or remove llms.txt file
        if (isset($_POST['llmscbt_llms_txt_enabled'])) {
            $content = sanitize_textarea_field($_POST['llmscbt_llms_txt_content']);
            $result = @file_put_contents(ABSPATH . 'llms.txt', $content);
            if ($result === false) {
                add_settings_error(
                    'llmscbt_messages',
                    'llmscbt_file_error',
                    __('Failed to create llms.txt file. Please check file permissions.', 'llms-central-ai-bot-tracker'),
                    'error'
                );
                return;
            }
        } else {
            if (file_exists(ABSPATH . 'llms.txt')) {
                @unlink(ABSPATH . 'llms.txt');
            }
        }
        
        add_settings_error(
            'llmscbt_messages',
            'llmscbt_message',
            __('Settings saved successfully!', 'llms-central-ai-bot-tracker'),
            'success'
        );
    }
}
