<?php
/**
 * Tracking Integration
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class LLMSCBT_Tracker {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Server-side bot detection (runs on EVERY request)
        add_action('init', array($this, 'track_bot_server_side'), 1);
        
        // Client-side widget (for browser tracking)
        add_action('wp_enqueue_scripts', array($this, 'enqueue_tracking_script'));
    }
    
    /**
     * Track AI bots server-side (before any output)
     */
    public function track_bot_server_side() {
        // Get user agent
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        
        if (empty($user_agent)) {
            return;
        }
        
        // AI bot patterns
        $bot_patterns = array(
            'GPTBot', 'ChatGPT', 'Claude', 'Anthropic', 'Google-Extended',
            'Googlebot', 'Bingbot', 'Applebot', 'PerplexityBot', 'YouBot',
            'Bytespider', 'Diffbot', 'Grokbot', 'X-Bot', 'cohere', 'Amazonbot'
        );
        
        // Check if it's a bot
        $is_bot = false;
        foreach ($bot_patterns as $pattern) {
            if (stripos($user_agent, $pattern) !== false) {
                $is_bot = true;
                break;
            }
        }
        
        if (!$is_bot) {
            return;
        }
        
        // Get domain from settings
        $domain = get_option('llmscbt_domain');
        if (empty($domain)) {
            return;
        }
        
        // Get current page
        $page = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '/';
        
        // Send to LLMS Central API (async, non-blocking)
        wp_remote_post(LLMSCBT_API_URL . '/api/bot-tracker', array(
            'blocking' => false, // Non-blocking (fire and forget)
            'timeout' => 1,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => $user_agent
            ),
            'body' => json_encode(array(
                'domain' => $domain,
                'page' => $page
            ))
        ));
    }
    
    /**
     * Enqueue tracking script
     */
    public function enqueue_tracking_script() {
        $domain = get_option('llmscbt_domain');
        
        if (empty($domain)) {
            return;
        }
        
        // Enqueue external tracking script
        wp_enqueue_script(
            'llmscbt-tracker',
            LLMSCBT_API_URL . '/widget.js',
            array(),
            LLMSCBT_VERSION,
            array(
                'in_footer' => true,
                'strategy' => 'defer'
            )
        );
        
        // Add data-domain attribute to script tag
        add_filter('script_loader_tag', array($this, 'add_domain_attribute'), 10, 3);
    }
    
    /**
     * Add data-domain attribute to tracking script
     */
    public function add_domain_attribute($tag, $handle, $src) {
        if ('llmscbt-tracker' !== $handle) {
            return $tag;
        }
        
        $domain = get_option('llmscbt_domain');
        if (empty($domain)) {
            return $tag;
        }
        
        // Add data-domain attribute
        return str_replace(' src=', ' data-domain="' . esc_attr($domain) . '" src=', $tag);
    }
}
