<?php
/**
 * Dashboard Widget
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class LLMSCBT_Widget {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_widget_styles'));
    }
    
    /**
     * Enqueue widget styles
     */
    public function enqueue_widget_styles($hook) {
        // Only load on dashboard
        if ('index.php' !== $hook) {
            return;
        }
        
        // Register a dummy handle for inline styles
        wp_register_style('llmscbt-widget', false);
        wp_enqueue_style('llmscbt-widget');
        
        // Add inline CSS
        $css = '
            .llms-central-widget-content { padding: 10px 0; }
            .llms-widget-stats { display: flex; gap: 15px; margin-bottom: 20px; }
            .stat-box { display: flex; align-items: center; gap: 10px; padding: 10px; background: #f0f0f1; border-radius: 4px; flex: 1; }
            .stat-box .dashicons { font-size: 24px; color: #2271b1; }
            .stat-info strong { display: block; font-size: 14px; }
            .stat-info p { margin: 0; font-size: 12px; color: #646970; }
            .llms-widget-bots h4 { margin: 15px 0 10px; }
            .bot-list { margin: 0; padding: 0; list-style: none; }
            .bot-list li { padding: 5px 0; font-size: 13px; }
            .bot-icon { margin-right: 5px; }
            .llms-widget-actions { margin-top: 15px; padding-top: 15px; border-top: 1px solid #dcdcde; }
            .llms-widget-actions .button { margin-right: 5px; }
        ';
        
        wp_add_inline_style('llmscbt-widget', $css);
    }
    
    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        wp_add_dashboard_widget(
            'llmscbt_widget',
            __('AI Bot Activity', 'llms-central-ai-bot-tracker'),
            array($this, 'render_widget')
        );
    }
    
    /**
     * Render widget content
     */
    public function render_widget() {
        $tracking_enabled = get_option('llmscbt_tracking_enabled');
        $consent_given = get_option('llmscbt_consent_given');
        $domain = get_option('llmscbt_domain');
        
        if (!$tracking_enabled || !$consent_given) {
            ?>
            <div class="llms-central-widget-inactive">
                <p>
                    <span class="dashicons dashicons-visibility" style="color: #999;"></span>
                    <?php esc_html_e('AI Bot Tracking is not active.', 'llms-central-ai-bot-tracker'); ?>
                </p>
                <p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=llmscbt')); ?>" class="button button-primary">
                        <?php esc_html_e('Enable Tracking', 'llms-central-ai-bot-tracker'); ?>
                    </a>
                </p>
            </div>
            <?php
            return;
        }
        
        ?>
        <div class="llms-central-widget-content">
            <div class="llms-widget-stats">
                <div class="stat-box">
                    <span class="dashicons dashicons-visibility"></span>
                    <div class="stat-info">
                        <strong><?php esc_html_e('Tracking Active', 'llms-central-ai-bot-tracker'); ?></strong>
                        <p><?php esc_html_e('Monitoring AI bot visits', 'llms-central-ai-bot-tracker'); ?></p>
                    </div>
                </div>
                
                <div class="stat-box">
                    <span class="dashicons dashicons-admin-site-alt3"></span>
                    <div class="stat-info">
                        <strong><?php echo esc_html($domain); ?></strong>
                        <p><?php esc_html_e('Your domain', 'llms-central-ai-bot-tracker'); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="llms-widget-bots">
                <h4><?php esc_html_e('Tracked AI Bots', 'llms-central-ai-bot-tracker'); ?></h4>
                <ul class="bot-list">
                    <li><span class="bot-icon">🤖</span> GPTBot (OpenAI)</li>
                    <li><span class="bot-icon">🤖</span> Claude (Anthropic)</li>
                    <li><span class="bot-icon">🤖</span> Google-Extended</li>
                    <li><span class="bot-icon">🤖</span> Perplexity</li>
                    <li><span class="bot-icon">🤖</span> +12 more AI crawlers</li>
                </ul>
            </div>
            
            <div class="llms-widget-actions">
                <a href="https://llmscentral.com/dashboard?domain=<?php echo esc_attr(urlencode($domain)); ?>" target="_blank" class="button button-primary">
                    <?php esc_html_e('View Full Analytics', 'llms-central-ai-bot-tracker'); ?>
                </a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=llmscbt')); ?>" class="button">
                    <?php esc_html_e('Settings', 'llms-central-ai-bot-tracker'); ?>
                </a>
            </div>
        </div>
        <?php
    }
}
