<?php
/**
 * Plugin Name: LLMS Central - AI Bot Tracker
 * Plugin URI: https://llmscentral.com/wordpress-plugin
 * Description: Track AI bots (GPT, Claude, Gemini) visiting your site and manage your llms.txt file. See which AI systems are crawling your content.
 * Version: 1.1.0
 * Author: LLMS Central
 * Author URI: https://llmscentral.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: llms-central-ai-bot-tracker
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LLMSCBT_VERSION', '1.1.0');
define('LLMSCBT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('LLMSCBT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LLMSCBT_API_URL', 'https://llmscentral.com');

/**
 * Main Plugin Class
 */
class LLMSCBT_Bot_Tracker {
    
    /**
     * Single instance of the class
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * Load required files
     */
    private function load_dependencies() {
        require_once LLMSCBT_PLUGIN_DIR . 'includes/class-admin.php';
        require_once LLMSCBT_PLUGIN_DIR . 'includes/class-tracker.php';
        require_once LLMSCBT_PLUGIN_DIR . 'includes/class-widget.php';
        require_once LLMSCBT_PLUGIN_DIR . 'includes/class-llms-txt.php';
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Activation/Deactivation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Initialize components
        add_action('plugins_loaded', array($this, 'init'));
        
        // Add settings link on plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $defaults = array(
            'tracking_enabled' => false,
            'domain' => parse_url(home_url(), PHP_URL_HOST),
            'llms_txt_enabled' => false,
            'llms_txt_content' => $this->get_default_llms_txt(),
            'consent_given' => false
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option('llmscbt_' . $key) === false) {
                add_option('llmscbt_' . $key, $value);
            }
        }
        
        // Create llms.txt file if enabled
        $this->maybe_create_llms_txt();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Remove llms.txt file
        $this->remove_llms_txt();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Initialize plugin components
     */
    public function init() {
        // Text domain is automatically loaded by WordPress for plugins hosted on WordPress.org
        
        // Initialize admin
        if (is_admin()) {
            LLMSCBT_Admin::get_instance();
            LLMSCBT_Widget::get_instance();
        }
        
        // Initialize tracker (frontend)
        if (get_option('llmscbt_tracking_enabled') && get_option('llmscbt_consent_given')) {
            LLMSCBT_Tracker::get_instance();
        }
        
        // Initialize llms.txt handler
        LLMSCBT_LLMS_TXT::get_instance();
    }
    
    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=llmscbt') . '">' . __('Settings', 'llms-central-ai-bot-tracker') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Get default llms.txt content
     */
    private function get_default_llms_txt() {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        
        return "# llms.txt - AI Training Guidelines for {$site_name}
# Generated by LLMS Central Bot Tracker
# Learn more: https://llmscentral.com

# General Policy
User-agent: *
Allow: /

# Contact Information
Contact: " . get_option('admin_email') . "
Website: {$site_url}

# Training Permissions
# Uncomment to restrict AI training:
# Disallow: /wp-content/uploads/
# Disallow: /premium-content/

# Attribution Required
Attribution: Required
Citation-format: {$site_name} - {$site_url}

# Commercial Use
Commercial-use: Allowed with attribution

# Last Updated: " . date('Y-m-d');
    }
    
    /**
     * Create llms.txt file
     */
    private function maybe_create_llms_txt() {
        if (get_option('llmscbt_llms_txt_enabled')) {
            $content = get_option('llmscbt_llms_txt_content');
            $file_path = ABSPATH . 'llms.txt';
            file_put_contents($file_path, $content);
        }
    }
    
    /**
     * Remove llms.txt file
     */
    private function remove_llms_txt() {
        $file_path = ABSPATH . 'llms.txt';
        if (file_exists($file_path)) {
            unlink($file_path);
        }
    }
}

/**
 * Initialize the plugin
 */
function llmscbt_init() {
    return LLMSCBT_Bot_Tracker::get_instance();
}

// Start the plugin
llmscbt_init();
